/** \file buff.c
 * implementation of the buff type, a buffer for arbitrary strings
 * and actually a superset of word_t
 *
 * \author David Relson <relson@osagesoftware.com>
 * \author Matthias Andree <matthias.andree@gmx.de> (buff_fgetsl)
 */

#include "common.h"

#include <stdbool.h>
#include <string.h>

#include "buff.h"
#include "fgetsl.h"
#include "xmalloc.h"

/* Function Definitions */
buff_t *buff_init(buff_t *self, byte *buff, uint used, uint size)
{
    self->t.u.text = buff;
    self->t.leng = used;
    self->read = 0;
    self->size = size;
    return self;
}

buff_t *buff_new(byte *buff, uint used, uint size)
{
    buff_t *self = (buff_t *)xmalloc(sizeof(buff_t));
    buff_init(self, buff, used, size);
    return self;
}

void buff_free(buff_t *self)
{
    xfree(self);
}

int buff_fgetsln(buff_t *self, FILE *in, uint maxlen)
{
    uint readpos = self->t.leng;
    int readcnt = xfgetsl((char *)self->t.u.text + readpos,
	    min(self->size - readpos, maxlen), in, true);
    /* WARNING: do not add NUL termination, the size must be exact! */
    self->read = readpos;
    if (readcnt >= 0)
	self->t.leng += readcnt;
    return readcnt;
}

int buff_add(buff_t *self, word_t *in)
{
    uint readpos = self->t.leng;
    int readcnt = in->leng;
    uint new_size = self->t.leng + in->leng;
    if (new_size > self->size) {
	self->t.u.text = (byte *)xrealloc(self->t.u.text, new_size + D);
	self->size = new_size;
    }
    self->read = readpos;
    self->t.leng += readcnt;
    memcpy(self->t.u.text + readpos, in->u.text, readcnt);
    Z(self->t.u.text[self->t.leng]);		/* for easier debugging - removable */

    return readcnt;
}

typedef void (func_word_puts)(const word_t *word_p, uint width, FILE *fp);
static void _buff_putf(const buff_t *self, uint width, FILE *fp, func_word_puts *func)
{
    word_t word;
    word.leng = self->t.leng - self->read;
    word.u.text = self->t.u.text + self->read;
    (*func)(&word, width, fp);
}

void buff_puts(const buff_t *self, uint width, FILE *fp)
{
    _buff_putf(self, width, fp, word_puts);
}

void buff_puts_escaped(const buff_t *self, uint width, FILE *fp)
{
    fprintf(fp, "<%lu>", (unsigned long)self->read);
    _buff_putf(self, width, fp, word_puts_escaped);
}

void buff_shift(buff_t *self, uint start, uint length)
{
    BOGO_ASSERT((unsigned long)start + length <= (unsigned long)self->t.leng,
		"Invalid buff_shift() parameters.");

    memmove(self->t.u.text + start, self->t.u.text + start + length, self->t.leng - start - length);

    /* reduce value in read according to erased content */
    if (self->read <= start+length && start <= self->t.leng) {
        self->read -= self->read >= start ? min(length, self->read - start) : 0;
    }
    self->t.leng -= length;
    Z(self->t.u.text[self->t.leng]);		/* for easier debugging - removable */
    return;
}
