import * as fs from 'node:fs';
import * as path from 'node:path';
import * as glob from 'glob';
import MarkdownIt from 'markdown-it';
import { githubSlugifier, resolveInternalDocumentLink, HrefKind, MdLinkKind, } from '@dsanders11/vscode-markdown-languageservice';
import { visit } from 'unist-util-visit';
import { fromMarkdown } from 'mdast-util-from-markdown';
import { Emitter } from 'vscode-languageserver';
import { TextDocument } from 'vscode-languageserver-textdocument';
import { URI } from 'vscode-uri';
// Helper function from `vscode-markdown-languageservice` codebase
function tryDecodeUri(str) {
    try {
        return decodeURI(str);
    }
    catch {
        return str;
    }
}
// Helper function from `vscode-markdown-languageservice` codebase
function createHref(sourceDocUri, link, workspace) {
    if (/^[a-z-][a-z-]+:/i.test(link)) {
        // Looks like a uri
        return { kind: HrefKind.External, uri: URI.parse(tryDecodeUri(link)) };
    }
    const resolved = resolveInternalDocumentLink(sourceDocUri, link, workspace);
    if (!resolved) {
        return undefined;
    }
    return {
        kind: HrefKind.Internal,
        path: resolved.resource,
        fragment: resolved.linkFragment,
    };
}
function positionToRange(position) {
    return {
        start: {
            character: position.start.column - 1,
            line: position.start.line - 1,
        },
        end: { character: position.end.column - 1, line: position.end.line - 1 },
    };
}
const mdIt = MarkdownIt({ html: true });
export class MarkdownParser {
    slugifier = githubSlugifier;
    async tokenize(document) {
        return mdIt.parse(document.getText(), {});
    }
}
export class DocsWorkspace {
    documentCache;
    root;
    globs;
    ignoreGlobs;
    constructor(root, globs, ignoreGlobs = []) {
        this.documentCache = new Map();
        this.root = root;
        this.globs = globs;
        this.ignoreGlobs = ignoreGlobs;
    }
    get workspaceFolders() {
        return [URI.file(this.root)];
    }
    async getAllMarkdownDocuments() {
        const files = this.globs.flatMap((pattern) => glob.sync(pattern, { ignore: this.ignoreGlobs, absolute: true, cwd: this.root }));
        for (const file of files) {
            const document = TextDocument.create(URI.file(file).toString(), 'markdown', 1, fs.readFileSync(file, 'utf8'));
            this.documentCache.set(file, document);
        }
        return this.documentCache.values();
    }
    hasMarkdownDocument(resource) {
        const relativePath = this.getWorkspaceRelativePath(resource);
        return (!relativePath.startsWith('..') &&
            !path.isAbsolute(relativePath) &&
            fs.existsSync(resource.fsPath));
    }
    getWorkspaceRelativePath(resource) {
        return path.relative(path.resolve(this.root), resource.fsPath);
    }
    async openMarkdownDocument(resource) {
        if (!this.documentCache.has(resource.fsPath)) {
            try {
                const document = TextDocument.create(resource.toString(), 'markdown', 1, fs.readFileSync(resource.fsPath, 'utf8'));
                this.documentCache.set(resource.fsPath, document);
            }
            catch {
                return undefined;
            }
        }
        return this.documentCache.get(resource.fsPath);
    }
    async stat(resource) {
        if (this.hasMarkdownDocument(resource)) {
            const stats = fs.statSync(resource.fsPath);
            return { isDirectory: stats.isDirectory() };
        }
        return undefined;
    }
    async readDirectory() {
        throw new Error('Not implemented');
    }
    //
    // These events are defined to fulfill the interface, but are never emitted
    // by this implementation since it's not meant for watching a workspace
    //
    #onDidChangeMarkdownDocument = new Emitter();
    onDidChangeMarkdownDocument = this.#onDidChangeMarkdownDocument.event;
    #onDidCreateMarkdownDocument = new Emitter();
    onDidCreateMarkdownDocument = this.#onDidCreateMarkdownDocument.event;
    #onDidDeleteMarkdownDocument = new Emitter();
    onDidDeleteMarkdownDocument = this.#onDidDeleteMarkdownDocument.event;
}
export class MarkdownLinkComputer {
    workspace;
    constructor(workspace) {
        this.workspace = workspace;
    }
    async getAllLinks(document) {
        const tree = fromMarkdown(document.getText());
        const links = [
            ...(await this.#getInlineLinks(document, tree)),
            ...(await this.#getReferenceLinks(document, tree)),
            ...(await this.#getLinkDefinitions(document, tree)),
        ];
        return links;
    }
    async #getInlineLinks(document, tree) {
        const documentUri = URI.parse(document.uri);
        const links = [];
        visit(tree, 'link', (node) => {
            const link = node;
            const href = createHref(documentUri, link.url, this.workspace);
            if (href) {
                const range = positionToRange(link.position);
                // NOTE - These haven't been implemented properly, but their
                //        values aren't used for the link linting use-case
                const targetRange = range;
                const hrefRange = range;
                const fragmentRange = undefined;
                links.push({
                    kind: MdLinkKind.Link,
                    href,
                    source: {
                        hrefText: link.url,
                        resource: documentUri,
                        range,
                        targetRange,
                        hrefRange,
                        fragmentRange,
                        pathText: link.url.split('#')[0],
                    },
                });
            }
        });
        return links;
    }
    async #getReferenceLinks(document, tree) {
        const links = [];
        visit(tree, ['imageReference', 'linkReference'], (node) => {
            const link = node;
            const range = positionToRange(link.position);
            // NOTE - These haven't been implemented properly, but their
            //        values aren't used for the link linting use-case
            const targetRange = range;
            const hrefRange = range;
            links.push({
                kind: MdLinkKind.Link,
                href: {
                    kind: HrefKind.Reference,
                    ref: link.label,
                },
                source: {
                    hrefText: link.label,
                    resource: URI.parse(document.uri),
                    range,
                    targetRange,
                    hrefRange,
                    fragmentRange: undefined,
                    pathText: link.label,
                },
            });
        });
        return links;
    }
    async #getLinkDefinitions(document, tree) {
        const documentUri = URI.parse(document.uri);
        const links = [];
        visit(tree, 'definition', (node) => {
            const definition = node;
            const href = createHref(documentUri, definition.url, this.workspace);
            if (href) {
                const range = positionToRange(definition.position);
                // NOTE - These haven't been implemented properly, but their
                //        values aren't used for the link linting use-case
                const targetRange = range;
                const hrefRange = range;
                const fragmentRange = undefined;
                links.push({
                    kind: MdLinkKind.Definition,
                    href,
                    ref: {
                        range,
                        text: definition.label,
                    },
                    source: {
                        hrefText: definition.url,
                        resource: documentUri,
                        range,
                        targetRange,
                        hrefRange,
                        fragmentRange,
                        pathText: definition.url.split('#')[0],
                    },
                });
            }
        });
        return links;
    }
}
export async function getCodeBlocks(content) {
    const tree = fromMarkdown(content);
    const codeBlocks = [];
    visit(tree, 'code', (node) => {
        codeBlocks.push(node);
    });
    return codeBlocks;
}
//# sourceMappingURL=markdown.js.map