import { SemVer } from 'semver';
export { SemVer };
import { Paths } from './paths';
export type SemOrStr = SemVer | string;
export interface ReleaseInfo {
    /** Electron version */
    version: string;
    /** Release date */
    date: string;
    /** Node.js version */
    node: string;
    /** V8 version */
    v8: string;
    /** uv version */
    uv: string;
    /** zlib version */
    zlib: string;
    /** OpenSSL version */
    openssl: string;
    /** Node.js modules version */
    modules: string;
    /** Chromium version */
    chrome: string;
    /** Files included in the release */
    files: Array<string>;
}
/**
 * Interface for an object that manages a list of Electron releases.
 *
 * See {@link BaseVersions} for testing situations.
 * See {@link ElectronVersions} for production.
 */
export interface Versions {
    /** Semver-Major numbers of branches that only have prereleases */
    readonly prereleaseMajors: number[];
    /** Semver-Major numbers of branches that have supported stable releases */
    readonly supportedMajors: number[];
    /** Semver-Major numbers of branches that are no longer supported */
    readonly obsoleteMajors: number[];
    /** The latest release (by version, not by date) */
    readonly latest: SemVer | undefined;
    /** The latest stable (by version, not by date) */
    readonly latestStable: SemVer | undefined;
    /** Full list of all known Electron releases, Sorted in branch order. */
    readonly versions: SemVer[];
    /** @returns true iff `version` is a release that this object knows about */
    isVersion(version: SemOrStr): boolean;
    /** @returns all versions matching that major number. Sorted in branch order. */
    inMajor(major: number): SemVer[];
    /** @returns all versions in a range, inclusive. Sorted in branch order. */
    inRange(a: SemOrStr, b: SemOrStr): SemVer[];
    /** @returns {@link ReleaseInfo} iff `version` is a release that this object knows about */
    getReleaseInfo(version: SemOrStr): ReleaseInfo | undefined;
}
export interface ElectronVersionsCreateOptions {
    /** Initial versions to use if there is no cache. When provided, no initial fetch is done */
    initialVersions?: unknown;
    /** Ignore the cache even if it exists and is fresh */
    ignoreCache?: boolean;
}
export declare function compareVersions(a: SemVer, b: SemVer): number;
/**
 * Implementation of {@link Versions} that does everything except self-populate.
 * It needs to be fed version info in its constructor.
 *
 * In production, use subclass '{@link ElectronVersions}'. This base class is
 * useful in testing because it's an easy way to inject fake test data into a
 * real Versions object.
 */
export declare class BaseVersions implements Versions {
    private readonly map;
    private readonly releaseInfo;
    protected setVersions(val: unknown): void;
    constructor(versions: unknown);
    get prereleaseMajors(): number[];
    get stableMajors(): number[];
    get supportedMajors(): number[];
    get obsoleteMajors(): number[];
    get versions(): SemVer[];
    get latest(): SemVer | undefined;
    get latestStable(): SemVer | undefined;
    isVersion(ver: SemOrStr): boolean;
    inMajor(major: number): SemVer[];
    inRange(a: SemOrStr, b: SemOrStr): SemVer[];
    getReleaseInfo(ver: SemOrStr): ReleaseInfo | undefined;
}
/**
 * Implementation of Versions that self-populates from release information at
 * https://releases.electronjs.org/releases.json .
 *
 * This is generally what to use in production.
 */
export declare class ElectronVersions extends BaseVersions {
    private readonly versionsCache;
    private mtimeMs;
    private constructor();
    private static fetchVersions;
    private static isCacheFresh;
    static create(paths?: Partial<Paths>, options?: ElectronVersionsCreateOptions): Promise<ElectronVersions>;
    fetch(): Promise<void>;
    private keepFresh;
    get prereleaseMajors(): number[];
    get stableMajors(): number[];
    get supportedMajors(): number[];
    get obsoleteMajors(): number[];
    get versions(): SemVer[];
    get latest(): SemVer | undefined;
    get latestStable(): SemVer | undefined;
    isVersion(ver: SemOrStr): boolean;
    inMajor(major: number): SemVer[];
    inRange(a: SemOrStr, b: SemOrStr): SemVer[];
}
