"use strict";
/*---------------------------------------------------------------------------------------------
 *  Copyright (c) Microsoft Corporation. All rights reserved.
 *  Licensed under the MIT License. See License.txt in the project root for license information.
 *--------------------------------------------------------------------------------------------*/
Object.defineProperty(exports, "__esModule", { value: true });
exports.MdRemoveLinkDefinitionCodeActionProvider = void 0;
const l10n = require("@vscode/l10n");
const lsp = require("vscode-languageserver-types");
const range_1 = require("../../types/range");
const textDocument_1 = require("../../types/textDocument");
const editBuilder_1 = require("../../util/editBuilder");
const diagnostics_1 = require("../diagnostics");
const util_1 = require("./util");
class MdRemoveLinkDefinitionCodeActionProvider {
    static #removeUnusedDefTitle = l10n.t('Remove unused link definition');
    static #removeDuplicateDefTitle = l10n.t('Remove duplicate link definition');
    *getActions(doc, range, context) {
        if (!this.#isEnabled(context)) {
            return;
        }
        const unusedDiagnosticLines = new Set();
        for (const diag of context.diagnostics) {
            if (diag.code === diagnostics_1.DiagnosticCode.link_unusedDefinition && diag.data && (0, range_1.rangeIntersects)(diag.range, range)) {
                const link = diag.data;
                yield this.#getRemoveDefinitionAction(doc, link, MdRemoveLinkDefinitionCodeActionProvider.#removeUnusedDefTitle);
                unusedDiagnosticLines.add(link.source.range.start.line);
            }
        }
        for (const diag of context.diagnostics) {
            if (diag.code === diagnostics_1.DiagnosticCode.link_duplicateDefinition && diag.data && (0, range_1.rangeIntersects)(diag.range, range)) {
                const link = diag.data;
                if (!unusedDiagnosticLines.has(link.source.range.start.line)) {
                    yield this.#getRemoveDefinitionAction(doc, link, MdRemoveLinkDefinitionCodeActionProvider.#removeDuplicateDefTitle);
                }
            }
        }
    }
    #isEnabled(context) {
        if (typeof context.only === 'undefined') {
            return true;
        }
        return context.only.some(kind => (0, util_1.codeActionKindContains)(lsp.CodeActionKind.QuickFix, kind));
    }
    #getRemoveDefinitionAction(doc, definition, title) {
        const builder = new editBuilder_1.WorkspaceEditBuilder();
        const range = definition.source.range;
        builder.replace((0, textDocument_1.getDocUri)(doc), (0, range_1.makeRange)(range.start.line, 0, range.start.line + 1, 0), '');
        return { title, kind: lsp.CodeActionKind.QuickFix, edit: builder.getEdit() };
    }
}
exports.MdRemoveLinkDefinitionCodeActionProvider = MdRemoveLinkDefinitionCodeActionProvider;
//# sourceMappingURL=removeLinkDefinition.js.map