"use strict";
var __createBinding = (this && this.__createBinding) || (Object.create ? (function(o, m, k, k2) {
    if (k2 === undefined) k2 = k;
    var desc = Object.getOwnPropertyDescriptor(m, k);
    if (!desc || ("get" in desc ? !m.__esModule : desc.writable || desc.configurable)) {
      desc = { enumerable: true, get: function() { return m[k]; } };
    }
    Object.defineProperty(o, k2, desc);
}) : (function(o, m, k, k2) {
    if (k2 === undefined) k2 = k;
    o[k2] = m[k];
}));
var __setModuleDefault = (this && this.__setModuleDefault) || (Object.create ? (function(o, v) {
    Object.defineProperty(o, "default", { enumerable: true, value: v });
}) : function(o, v) {
    o["default"] = v;
});
var __importStar = (this && this.__importStar) || function (mod) {
    if (mod && mod.__esModule) return mod;
    var result = {};
    if (mod != null) for (var k in mod) if (k !== "default" && Object.prototype.hasOwnProperty.call(mod, k)) __createBinding(result, mod, k);
    __setModuleDefault(result, mod);
    return result;
};
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.Runner = void 0;
const child_process_1 = require("child_process");
const fs = __importStar(require("fs"));
const os = __importStar(require("os"));
const path = __importStar(require("path"));
const debug_1 = __importDefault(require("debug"));
const getos_1 = __importDefault(require("getos"));
const semver_1 = require("semver");
const util_1 = require("util");
const installer_1 = require("./installer");
const versions_1 = require("./versions");
const fiddle_1 = require("./fiddle");
const paths_1 = require("./paths");
const DefaultRunnerOpts = {
    args: [],
    headless: false,
    out: process.stdout,
    showConfig: true,
};
class Runner {
    constructor(installer, versions, fiddleFactory) {
        this.installer = installer;
        this.versions = versions;
        this.fiddleFactory = fiddleFactory;
        this.osInfo = '';
        // FIXME(anyone): minor wart, 'source' is incorrect here if it's a local build
        this.spawnInfo = (version, exec, fiddle) => [
            '',
            '🧪 Testing',
            '',
            `  - date: ${new Date().toISOString()}`,
            '',
            '  - fiddle:',
            `      - source: ${fiddle.source}`,
            `      - local copy: ${path.dirname(fiddle.mainPath)}`,
            '',
            `  - electron_version: ${version}`,
            `      - source: https://github.com/electron/electron/releases/tag/v${version}`,
            `      - local copy: ${path.dirname(exec)}`,
            '',
            '  - test platform:',
            `      - os_arch: ${os.arch()}`,
            `      - os_platform: ${process.platform}`,
            `      - os_release: ${os.release()}`,
            `      - os_version: ${os.version()}`,
            `      - getos: ${this.osInfo}`,
            '',
        ].join('\n');
        (0, getos_1.default)((err, result) => (this.osInfo = (0, util_1.inspect)(result || err)));
    }
    static async create(opts = {}) {
        const paths = Object.freeze({ ...paths_1.DefaultPaths, ...(opts.paths || {}) });
        const installer = opts.installer || new installer_1.Installer(paths);
        const versions = opts.versions || (await versions_1.ElectronVersions.create(paths));
        const factory = opts.fiddleFactory || new fiddle_1.FiddleFactory(paths.fiddles);
        return new Runner(installer, versions, factory);
    }
    /**
     * Figure out how to run the user-specified `electron` value.
     *
     * - if it's an existing directory, look for an execPath in it.
     * - if it's an existing file, run it. It's a local build.
     * - if it's a version number, delegate to the installer
     *
     * @param val - a version number, directory, or executable
     * @returns a path to an Electron executable
     */
    async getExec(electron) {
        try {
            const stat = fs.statSync(electron);
            // if it's on the filesystem but not a directory, use it directly
            if (!stat.isDirectory())
                return electron;
            // if it's on the filesystem as a directory, look for execPath
            const name = installer_1.Installer.getExecPath(electron);
            if (fs.existsSync(name))
                return name;
        }
        catch (_a) {
            // if it's a version, install it
            if (this.versions.isVersion(electron))
                return await this.installer.install(electron);
        }
        throw new Error(`Unrecognized electron name: "${electron}"`);
    }
    /** If headless specified on  *nix, try to run with xvfb-run */
    static headless(exec, args) {
        if (process.platform !== 'darwin' && process.platform !== 'win32') {
            // try to get a free server number
            args.unshift('--auto-servernum', exec);
            exec = 'xvfb-run';
        }
        return { exec, args };
    }
    async spawn(versionIn, fiddleIn, opts = {}) {
        var _a, _b;
        const d = (0, debug_1.default)('fiddle-core:Runner.spawn');
        // process the input parameters
        opts = { ...DefaultRunnerOpts, ...opts };
        const version = versionIn instanceof semver_1.SemVer ? versionIn.version : versionIn;
        const fiddle = await this.fiddleFactory.create(fiddleIn);
        if (!fiddle)
            throw new Error(`Invalid fiddle: "${(0, util_1.inspect)(fiddleIn)}"`);
        // set up the electron binary and the fiddle
        const electronExec = await this.getExec(version);
        let exec = electronExec;
        let args = [...(opts.args || []), fiddle.mainPath];
        if (opts.headless)
            ({ exec, args } = Runner.headless(exec, args));
        if (opts.out && opts.showConfig) {
            opts.out.write(`${this.spawnInfo(version, electronExec, fiddle)}\n`);
        }
        d((0, util_1.inspect)({ exec, args, opts }));
        const child = (0, child_process_1.spawn)(exec, args, opts);
        if (opts.out) {
            (_a = child.stdout) === null || _a === void 0 ? void 0 : _a.pipe(opts.out);
            (_b = child.stderr) === null || _b === void 0 ? void 0 : _b.pipe(opts.out);
        }
        return child;
    }
    static displayEmoji(result) {
        switch (result.status) {
            case 'system_error':
                return '🟠';
            case 'test_error':
                return '🔵';
            case 'test_failed':
                return '🔴';
            case 'test_passed':
                return '🟢';
        }
    }
    static displayResult(result) {
        const text = Runner.displayEmoji(result);
        switch (result.status) {
            case 'system_error':
                return text + ' system error: test did not pass or fail';
            case 'test_error':
                return text + ' test error: test did not pass or fail';
            case 'test_failed':
                return text + ' failed';
            case 'test_passed':
                return text + ' passed';
        }
    }
    async run(version, fiddle, opts = DefaultRunnerOpts) {
        const subprocess = await this.spawn(version, fiddle, opts);
        return new Promise((resolve) => {
            subprocess.on('error', () => {
                return resolve({ status: 'system_error' });
            });
            subprocess.on('exit', (code) => {
                if (code === 0)
                    return resolve({ status: 'test_passed' });
                if (code === 1)
                    return resolve({ status: 'test_failed' });
                return resolve({ status: 'test_error' });
            });
        });
    }
    async bisect(version_a, version_b, fiddleIn, opts = DefaultRunnerOpts) {
        const { out } = opts;
        const log = (first, ...rest) => {
            if (out) {
                out.write([first, ...rest].join(' '));
                out.write('\n');
            }
        };
        const versions = this.versions.inRange(version_a, version_b);
        const fiddle = await this.fiddleFactory.create(fiddleIn);
        if (!fiddle)
            throw new Error(`Invalid fiddle: "${(0, util_1.inspect)(fiddleIn)}"`);
        const displayIndex = (i) => '#' + i.toString().padStart(4, ' ');
        log([
            '📐 Bisect Requested',
            '',
            ` - gist is ${fiddle.source}`,
            // eslint-disable-next-line @typescript-eslint/no-base-to-string
            ` - the version range is [${version_a.toString()}..${version_b.toString()}]`,
            ` - there are ${versions.length} versions in this range:`,
            '',
            ...versions.map((ver, i) => `${displayIndex(i)} - ${ver.version}`),
        ].join('\n'));
        // bisect through the releases
        const LEFT_POS = 0;
        const RIGHT_POS = versions.length - 1;
        let left = LEFT_POS;
        let right = RIGHT_POS;
        let result = undefined;
        const testOrder = [];
        const results = new Array(versions.length);
        while (left + 1 < right) {
            const mid = Math.round(left + (right - left) / 2);
            const ver = versions[mid];
            testOrder.push(mid);
            log(`bisecting, range [${left}..${right}], mid ${mid} (${ver.version})`);
            result = await this.run(ver.version, fiddle, opts);
            results[mid] = result;
            log(`${Runner.displayResult(result)} ${versions[mid].version}\n`);
            if (result.status === 'test_passed') {
                left = mid;
                continue;
            }
            else if (result.status === 'test_failed') {
                right = mid;
                continue;
            }
            else {
                break;
            }
        }
        // validates the status of the boundary versions if we've reached the end
        // of the bisect and one of our pointers is at a boundary.
        const boundaries = [];
        if (left === LEFT_POS && !results[LEFT_POS])
            boundaries.push(LEFT_POS);
        if (right === RIGHT_POS && !results[RIGHT_POS])
            boundaries.push(RIGHT_POS);
        for (const position of boundaries) {
            const result = await this.run(versions[position].version, fiddle, opts);
            results[position] = result;
            log(`${Runner.displayResult(result)} ${versions[position].version}\n`);
        }
        log(`🏁 finished bisecting across ${versions.length} versions...`);
        versions.forEach((ver, i) => {
            const n = testOrder.indexOf(i);
            if (n === -1)
                return;
            log(displayIndex(i), Runner.displayResult(results[i]), ver, `(test #${n + 1})`);
        });
        log('\n🏁 Done bisecting');
        const success = results[left].status === 'test_passed' &&
            results[right].status === 'test_failed';
        if (success) {
            const good = versions[left].version;
            const bad = versions[right].version;
            log([
                `${Runner.displayResult(results[left])} ${good}`,
                `${Runner.displayResult(results[right])} ${bad}`,
                'Commits between versions:',
                `https://github.com/electron/electron/compare/v${good}...v${bad} ↔`,
            ].join('\n'));
            return {
                range: [versions[left].version, versions[right].version],
                status: 'bisect_succeeded',
            };
        }
        else {
            // FIXME: log some failure
            if ((result === null || result === void 0 ? void 0 : result.status) === 'test_error' ||
                (result === null || result === void 0 ? void 0 : result.status) === 'system_error') {
                return { status: result.status };
            }
            if (results[left].status === results[right].status) {
                return { status: 'test_error' };
            }
            return { status: 'system_error' };
        }
    }
}
exports.Runner = Runner;
