import { DocumentationTag, } from '@electron/docs-parser';
import chalk from 'chalk';
import d from 'debug';
import _ from 'lodash';
import * as utils from './utils.js';
const debug = d('dynamic-param');
// Object of interfaces we need to declare
const paramInterfacesToDeclare = {};
// Interfaces that we would declare with these prefixes should remove them before declaration
const impoliteInterfaceNames = ['Get', 'Set', 'Show'];
const polite = (s) => {
    for (let i = 0; i < impoliteInterfaceNames.length; i++) {
        if (s.startsWith(impoliteInterfaceNames[i]))
            return polite(s.substring(impoliteInterfaceNames[i].length));
    }
    return s;
};
// Ignore descriptions when comparing objects
const ignoreDescriptions = (props) => _.map(props, (p) => {
    const { description, ...toReturn } = p;
    return toReturn;
}).sort((a, b) => a.name.localeCompare(b.name));
const noDescriptionCache = new WeakMap();
const unsetDescriptions = (o) => {
    if (noDescriptionCache.has(o))
        return noDescriptionCache.get(o);
    if (typeof o !== 'object' || !o)
        return o;
    const val = Array.isArray(o)
        ? o.map((item) => unsetDescriptions(item))
        : Object.keys(o).reduce((accum, key) => {
            if (key === 'description')
                return accum;
            accum[key] = unsetDescriptions(o[key]);
            return accum;
        }, {});
    noDescriptionCache.set(o, val);
    return val;
};
// Given a parameter create a new interface and return it's name + array modifier
// IName is the proposed interface name prefix
// backupIName is a slightly longer IName in case IName is already taken
const createParamInterface = (param, IName = '', backupIName = '', finalBackupIName = '') => {
    const maybeArray = (type) => (param.collection ? `Array<${type}>` : type);
    const potentialExistingArgType = polite(IName);
    const potentialExistingArgName = _.lowerFirst(polite(IName));
    let argType = polite(IName) + _.upperFirst(_.camelCase(param.name));
    let argName = param.name;
    // TODO: Note.  It is still possible for even backupIName to be already used
    let usingExistingParamInterface = false;
    _.forIn(paramInterfacesToDeclare, (value, key) => {
        const test = unsetDescriptions(_.assign({}, param, {
            name: argName,
            tName: argType,
            required: value.required,
            additionalTags: param.additionalTags || [],
        }));
        const potentialTest = unsetDescriptions(_.assign({}, param, {
            name: potentialExistingArgName,
            tName: potentialExistingArgType,
            required: value.required,
            additionalTags: param.additionalTags || [],
        }));
        const unsetValue = unsetDescriptions(value);
        if (_.isEqual(test, unsetValue) || _.isEqual(potentialTest, unsetValue)) {
            usingExistingParamInterface = true;
            debug(chalk.cyan(`Using existing type for param name ${argType} --> ${key} in Interface: ${_.upperFirst(param.tName)} --- This is because their structure is identical`));
            argType = key;
            return false;
        }
    });
    if (usingExistingParamInterface) {
        return maybeArray(argType);
    }
    if (paramInterfacesToDeclare[argType] &&
        !_.isEqual(ignoreDescriptions(paramInterfacesToDeclare[argType].properties), ignoreDescriptions(param.properties))) {
        if (backupIName) {
            return createParamInterface(param, backupIName, finalBackupIName);
        }
        console.error(argType, IName, backupIName, finalBackupIName, ignoreDescriptions(paramInterfacesToDeclare[argType].properties), '\n', ignoreDescriptions(param.properties));
        throw Error(`Interface "${argType}" has already been declared`);
    }
    // Update the params interfaces we still have to define
    paramInterfacesToDeclare[argType] = param;
    paramInterfacesToDeclare[argType].name = argName;
    paramInterfacesToDeclare[argType].tName = argType;
    return maybeArray(argType);
};
const flushParamInterfaces = (API, addToOutput) => {
    const declared = {};
    while (Object.keys(paramInterfacesToDeclare).length > 0) {
        const nestedInterfacesToDeclare = {};
        Object.keys(paramInterfacesToDeclare)
            .sort((a, b) => paramInterfacesToDeclare[a].tName.localeCompare(paramInterfacesToDeclare[b].tName))
            .forEach((paramKey) => {
            if (paramKey === 'Event') {
                throw 'Unexpected dynamic Event type, should be routed through the Event handler';
            }
            if (declared[paramKey]) {
                const toDeclareCheck = Object.assign({}, paramInterfacesToDeclare[paramKey]);
                const declaredCheck = Object.assign({}, declared[paramKey]);
                for (const prop of ['type', 'collection', 'required', 'description']) {
                    delete toDeclareCheck[prop];
                    delete declaredCheck[prop];
                }
                if (!_.isEqual(toDeclareCheck, declaredCheck)) {
                    throw new Error('Ruh roh, "' + paramKey + '" is already declared');
                }
                delete paramInterfacesToDeclare[paramKey];
                return;
            }
            declared[paramKey] = paramInterfacesToDeclare[paramKey];
            const param = paramInterfacesToDeclare[paramKey];
            const paramAPI = [];
            paramAPI.push(`interface ${_.upperFirst(param.tName)}${param.extends ? ` extends ${param.extends}` : ''} {`);
            param.properties = param.properties || [];
            param.properties.forEach((paramProperty) => {
                if (paramProperty.description) {
                    utils.extendArray(paramAPI, utils.wrapComment(paramProperty.description, paramProperty.additionalTags));
                }
                if (!Array.isArray(paramProperty.type) && paramProperty.type.toLowerCase() === 'object') {
                    let argType = paramProperty.__type || _.upperFirst(_.camelCase(paramProperty.name));
                    if (API.some((a) => a.name === argType)) {
                        paramProperty.type = argType;
                        debug(chalk.red(`Auto-correcting type from Object --> ${argType} in Interface: ${_.upperFirst(param.tName)} --- This should be fixed in the docs`));
                    }
                    else {
                        nestedInterfacesToDeclare[argType] = paramProperty;
                        nestedInterfacesToDeclare[argType].name = paramProperty.name;
                        nestedInterfacesToDeclare[argType].tName = argType;
                        paramProperty.type = argType;
                    }
                }
                if (Array.isArray(paramProperty.type)) {
                    paramProperty.type = paramProperty.type.map((paramPropertyType) => {
                        const functionProp = paramPropertyType;
                        if (paramPropertyType.type === 'Function' && functionProp.parameters) {
                            return {
                                ...paramPropertyType,
                                // FIXME: functionProp should slot in here perfectly
                                type: utils.genMethodString(DynamicParamInterfaces, param, functionProp, true),
                            };
                        }
                        else if (typeof paramPropertyType.type === 'string' &&
                            paramPropertyType.type.toLowerCase() === 'object') {
                            let argType = paramProperty.__type || _.upperFirst(_.camelCase(paramProperty.name));
                            if (API.some((a) => a.name === argType)) {
                                paramPropertyType.type = argType;
                                debug(chalk.red(`Auto-correcting type from Object --> ${argType} in Interface: ${_.upperFirst(param.tName)} --- This should be fixed in the docs`));
                            }
                            else {
                                nestedInterfacesToDeclare[argType] = paramPropertyType;
                                nestedInterfacesToDeclare[argType].name = paramProperty.name;
                                nestedInterfacesToDeclare[argType].tName = argType;
                                paramPropertyType.type = argType;
                            }
                        }
                        return paramPropertyType;
                    });
                }
                const isReadonly = (paramProperty.additionalTags || []).includes(DocumentationTag.AVAILABILITY_READONLY)
                    ? 'readonly '
                    : '';
                if (!Array.isArray(paramProperty.type) &&
                    paramProperty.type.toLowerCase() === 'function') {
                    // FIXME: functionProp should slot in here perfectly
                    paramAPI.push(`${isReadonly}${paramProperty.name}${utils.isOptional(paramProperty) ? '?' : ''}: ${utils.genMethodString(DynamicParamInterfaces, param, paramProperty, true)};`);
                }
                else {
                    paramAPI.push(`${isReadonly}${paramProperty.name}${utils.isOptional(paramProperty) ? '?' : ''}: ${utils.typify(paramProperty)};`);
                }
            });
            paramAPI.push('}');
            addToOutput(paramAPI.map((l, index) => (index === 0 || index === paramAPI.length - 1 ? l : `  ${l}`)));
            delete paramInterfacesToDeclare[paramKey];
        });
        Object.assign(paramInterfacesToDeclare, nestedInterfacesToDeclare);
    }
    return Object.keys(declared);
};
export class DynamicParamInterfaces {
    static createParamInterface = createParamInterface;
    static flushParamInterfaces = flushParamInterfaces;
}
utils.setParamInterfaces(DynamicParamInterfaces);
//# sourceMappingURL=dynamic-param-interfaces.js.map