"use strict";
var __createBinding = (this && this.__createBinding) || (Object.create ? (function(o, m, k, k2) {
    if (k2 === undefined) k2 = k;
    var desc = Object.getOwnPropertyDescriptor(m, k);
    if (!desc || ("get" in desc ? !m.__esModule : desc.writable || desc.configurable)) {
      desc = { enumerable: true, get: function() { return m[k]; } };
    }
    Object.defineProperty(o, k2, desc);
}) : (function(o, m, k, k2) {
    if (k2 === undefined) k2 = k;
    o[k2] = m[k];
}));
var __setModuleDefault = (this && this.__setModuleDefault) || (Object.create ? (function(o, v) {
    Object.defineProperty(o, "default", { enumerable: true, value: v });
}) : function(o, v) {
    o["default"] = v;
});
var __importStar = (this && this.__importStar) || function (mod) {
    if (mod && mod.__esModule) return mod;
    var result = {};
    if (mod != null) for (var k in mod) if (k !== "default" && Object.prototype.hasOwnProperty.call(mod, k)) __createBinding(result, mod, k);
    __setModuleDefault(result, mod);
    return result;
};
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.Installer = exports.InstallState = void 0;
const fs = __importStar(require("fs-extra"));
const path = __importStar(require("path"));
const semver_1 = __importDefault(require("semver"));
const debug_1 = __importDefault(require("debug"));
const extract_zip_1 = __importDefault(require("extract-zip"));
const events_1 = require("events");
const rimraf_1 = require("rimraf");
const get_1 = require("@electron/get");
const util_1 = require("util");
const paths_1 = require("./paths");
function getZipName(version) {
    return `electron-v${version}-${process.platform}-${process.arch}.zip`;
}
/**
 * The state of a current Electron version.
 * See {@link Installer.state} to get this value.
 * See Installer.on('state-changed') to watch for state changes.
 */
var InstallState;
(function (InstallState) {
    InstallState["missing"] = "missing";
    InstallState["downloading"] = "downloading";
    InstallState["downloaded"] = "downloaded";
    InstallState["installing"] = "installing";
    InstallState["installed"] = "installed";
})(InstallState = exports.InstallState || (exports.InstallState = {}));
/**
 * Manage downloading and installing Electron versions.
 *
 * An Electron release's .zip is downloaded into `paths.electronDownloads`,
 * which holds all the downloaded zips.
 *
 * The installed version is unzipped into `paths.electronInstall`. Only one
 * version is installed at a time -- installing a new version overwrites the
 * current one in `paths.electronInstall`.
 *
 * See {@link DefaultPaths} for the default paths.
 */
class Installer extends events_1.EventEmitter {
    constructor(pathsIn = {}) {
        super();
        this.stateMap = new Map();
        /** map of version string to currently-running active Promise */
        this.downloading = new Map();
        /** keep a track of all currently installing versions */
        this.installing = new Set();
        this.paths = Object.freeze({ ...paths_1.DefaultPaths, ...pathsIn });
        this.rebuildStates();
    }
    static execSubpath(platform = process.platform) {
        switch (platform) {
            case 'darwin':
                return 'Electron.app/Contents/MacOS/Electron';
            case 'win32':
                return 'electron.exe';
            default:
                return 'electron';
        }
    }
    static getExecPath(folder) {
        return path.join(folder, Installer.execSubpath());
    }
    state(version) {
        return this.stateMap.get(version) || InstallState.missing;
    }
    setState(version, state) {
        const d = (0, debug_1.default)('fiddle-core:Installer:setState');
        const oldState = this.state(version);
        if (state === InstallState.missing) {
            this.stateMap.delete(version);
        }
        else {
            this.stateMap.set(version, state);
        }
        const newState = this.state(version);
        d((0, util_1.inspect)({ version, oldState, newState }));
        if (oldState !== newState) {
            const event = { version, state: newState };
            d('emitting state-changed', (0, util_1.inspect)(event));
            this.emit('state-changed', event);
        }
    }
    rebuildStates() {
        this.stateMap.clear();
        // currently installed...
        try {
            const versionFile = path.join(this.paths.electronInstall, 'version');
            const version = fs.readFileSync(versionFile, 'utf8').trim();
            this.setState(version, InstallState.installed);
        }
        catch (_a) {
            // no current version
        }
        this.installing.forEach((version) => {
            this.setState(version, InstallState.installing);
        });
        // already downloaded...
        const str = `^electron-v(.*)-${process.platform}-${process.arch}.zip$`;
        const reg = new RegExp(str);
        try {
            for (const file of fs.readdirSync(this.paths.electronDownloads)) {
                const match = reg.exec(file);
                if (match) {
                    this.setState(match[1], InstallState.downloaded);
                }
                else {
                    // Case when the download path already has the unzipped electron version
                    const versionFile = path.join(this.paths.electronDownloads, file, 'version');
                    if (fs.existsSync(versionFile)) {
                        const version = fs.readFileSync(versionFile, 'utf8').trim();
                        if (semver_1.default.valid(version)) {
                            this.setState(version, InstallState.downloaded);
                        }
                    }
                }
            }
        }
        catch (_b) {
            // no download directory yet
        }
        // being downloaded now...
        for (const version of this.downloading.keys()) {
            this.setState(version, InstallState.downloading);
        }
    }
    /** Removes an Electron download or Electron install from the disk. */
    async remove(version) {
        const d = (0, debug_1.default)('fiddle-core:Installer:remove');
        d(version);
        let isBinaryDeleted = false;
        // utility to re-run removal functions upon failure
        // due to windows filesystem lockfile jank
        const rerunner = async (path, func, counter = 1) => {
            try {
                func(path);
                return true;
            }
            catch (error) {
                console.warn(`Installer: failed to run ${func.name} for ${version}, but failed`, error);
                if (counter < 4) {
                    console.log(`Installer: Trying again to run ${func.name}`);
                    await rerunner(path, func, counter + 1);
                }
            }
            return false;
        };
        const binaryCleaner = (path) => {
            if (fs.existsSync(path)) {
                const { noAsar } = process;
                try {
                    process.noAsar = true;
                    fs.removeSync(path);
                }
                finally {
                    process.noAsar = noAsar;
                }
            }
        };
        // get the zip path
        const zipPath = path.join(this.paths.electronDownloads, getZipName(version));
        // Or, maybe the version was already installed and kept in file system
        const preInstalledPath = path.join(this.paths.electronDownloads, version);
        const isZipDeleted = await rerunner(zipPath, binaryCleaner);
        const isPathDeleted = await rerunner(preInstalledPath, binaryCleaner);
        // maybe uninstall it
        if (this.installedVersion === version) {
            isBinaryDeleted = await rerunner(this.paths.electronInstall, binaryCleaner);
        }
        else {
            // If the current version binary doesn't exists
            isBinaryDeleted = true;
        }
        if ((isZipDeleted || isPathDeleted) && isBinaryDeleted) {
            this.setState(version, InstallState.missing);
        }
        else {
            // Ideally the execution shouldn't reach this point
            console.warn(`Installer: Failed to remove version ${version}`);
        }
    }
    /** The current Electron installation, if any. */
    get installedVersion() {
        for (const [version, state] of this.stateMap)
            if (state === InstallState.installed)
                return version;
    }
    async download(version, opts) {
        var _a, _b;
        let pctDone = 0;
        const getProgressCallback = (progress) => {
            if (opts === null || opts === void 0 ? void 0 : opts.progressCallback) {
                // Call the user passed callback function
                opts.progressCallback(progress);
            }
            const pct = Math.round(progress.percent * 100);
            if (pctDone + 10 <= pct) {
                const emoji = pct >= 100 ? '🏁' : '⏳';
                // FIXME(anyone): is there a better place than console.log for this?
                console.log(`${emoji} downloading ${version} - ${pct}%`);
                pctDone = pct;
            }
        };
        const zipFile = await (0, get_1.download)(version, {
            mirrorOptions: {
                mirror: (_a = opts === null || opts === void 0 ? void 0 : opts.mirror) === null || _a === void 0 ? void 0 : _a.electronMirror,
                nightlyMirror: (_b = opts === null || opts === void 0 ? void 0 : opts.mirror) === null || _b === void 0 ? void 0 : _b.electronNightlyMirror,
            },
            downloadOptions: {
                quiet: true,
                getProgressCallback,
            },
        });
        return zipFile;
    }
    async ensureDownloadedImpl(version, opts) {
        const d = (0, debug_1.default)(`fiddle-core:Installer:${version}:ensureDownloadedImpl`);
        const { electronDownloads } = this.paths;
        const zipFile = path.join(electronDownloads, getZipName(version));
        const zipFileExists = fs.existsSync(zipFile);
        const state = this.state(version);
        if (state === InstallState.downloaded) {
            const preInstalledPath = path.join(electronDownloads, version);
            if (!zipFileExists && fs.existsSync(preInstalledPath)) {
                return {
                    path: preInstalledPath,
                    alreadyExtracted: true,
                };
            }
        }
        if (state === InstallState.missing || !zipFileExists) {
            d(`"${zipFile}" does not exist; downloading now`);
            this.setState(version, InstallState.downloading);
            try {
                const tempFile = await this.download(version, opts);
                await fs.ensureDir(electronDownloads);
                await fs.move(tempFile, zipFile);
            }
            catch (err) {
                this.setState(version, InstallState.missing);
                throw err;
            }
            this.setState(version, InstallState.downloaded);
            d(`"${zipFile}" downloaded`);
        }
        else {
            d(`"${zipFile}" exists; no need to download`);
        }
        return {
            path: zipFile,
            alreadyExtracted: false,
        };
    }
    async ensureDownloaded(version, opts) {
        const { downloading: promises } = this;
        let promise = promises.get(version);
        if (promise)
            return promise;
        promise = this.ensureDownloadedImpl(version, opts).finally(() => promises.delete(version));
        promises.set(version, promise);
        return promise;
    }
    async install(version, opts) {
        const d = (0, debug_1.default)(`fiddle-core:Installer:${version}:install`);
        const { electronInstall } = this.paths;
        const isVersionInstalling = this.installing.has(version);
        const electronExec = Installer.getExecPath(electronInstall);
        if (isVersionInstalling) {
            throw new Error(`Currently installing "${version}"`);
        }
        this.installing.add(version);
        try {
            // see if the current version (if any) is already `version`
            const { installedVersion } = this;
            if (installedVersion === version) {
                d(`already installed`);
            }
            else {
                const { path: source, alreadyExtracted } = await this.ensureDownloaded(version, opts);
                // An unzipped version already exists at `electronDownload` path
                if (alreadyExtracted) {
                    await this.installVersionImpl(version, source, () => {
                        // Simply copy over the files from preinstalled version to `electronInstall`
                        const { noAsar } = process;
                        process.noAsar = true;
                        fs.copySync(source, electronInstall);
                        process.noAsar = noAsar;
                    });
                }
                else {
                    await this.installVersionImpl(version, source, async () => {
                        // FIXME(anyone) is there a less awful way to wrangle asar
                        const { noAsar } = process;
                        try {
                            process.noAsar = true;
                            await (0, extract_zip_1.default)(source, { dir: electronInstall });
                        }
                        finally {
                            process.noAsar = noAsar;
                        }
                    });
                }
            }
        }
        finally {
            this.installing.delete(version);
        }
        // return the full path to the electron executable
        d((0, util_1.inspect)({ electronExec, version }));
        return electronExec;
    }
    async installVersionImpl(version, source, installCallback) {
        const { paths: { electronInstall }, installedVersion, } = this;
        const d = (0, debug_1.default)(`fiddle-core:Installer:${version}:install`);
        const originalState = this.state(version);
        this.setState(version, InstallState.installing);
        try {
            d(`installing from "${source}"`);
            const { noAsar } = process;
            try {
                process.noAsar = true;
                (0, rimraf_1.rimrafSync)(electronInstall);
            }
            finally {
                process.noAsar = noAsar;
            }
            // Call the user defined callback which unzips/copies files content
            if (installCallback) {
                await installCallback();
            }
        }
        catch (err) {
            this.setState(version, originalState);
            throw err;
        }
        if (installedVersion) {
            this.setState(installedVersion, InstallState.downloaded);
        }
        this.setState(version, InstallState.installed);
    }
}
exports.Installer = Installer;
