'use strict';

Object.defineProperty(exports, '__esModule', { value: true });

function _interopDefault (ex) { return (ex && (typeof ex === 'object') && 'default' in ex) ? ex['default'] : ex; }

var oauthAuthorizationUrl = require('@octokit/oauth-authorization-url');
var request = require('@octokit/request');
var requestError = require('@octokit/request-error');
var btoa = _interopDefault(require('btoa-lite'));

const VERSION = "2.0.4";

function requestToOAuthBaseUrl(request) {
  const endpointDefaults = request.endpoint.DEFAULTS;
  return /^https:\/\/(api\.)?github\.com$/.test(endpointDefaults.baseUrl) ? "https://github.com" : endpointDefaults.baseUrl.replace("/api/v3", "");
}
async function oauthRequest(request, route, parameters) {
  const withOAuthParameters = {
    baseUrl: requestToOAuthBaseUrl(request),
    headers: {
      accept: "application/json"
    },
    ...parameters
  };
  const response = await request(route, withOAuthParameters);
  if ("error" in response.data) {
    const error = new requestError.RequestError(`${response.data.error_description} (${response.data.error}, ${response.data.error_uri})`, 400, {
      request: request.endpoint.merge(route, withOAuthParameters),
      headers: response.headers
    });
    // @ts-ignore add custom response property until https://github.com/octokit/request-error.js/issues/169 is resolved
    error.response = response;
    throw error;
  }
  return response;
}

function getWebFlowAuthorizationUrl({
  request: request$1 = request.request,
  ...options
}) {
  const baseUrl = requestToOAuthBaseUrl(request$1);
  // @ts-expect-error TypeScript wants `clientType` to be set explicitly ¯\_(ツ)_/¯
  return oauthAuthorizationUrl.oauthAuthorizationUrl({
    ...options,
    baseUrl
  });
}

async function exchangeWebFlowCode(options) {
  const request$1 = options.request || /* istanbul ignore next: we always pass a custom request in tests */
  request.request;
  const response = await oauthRequest(request$1, "POST /login/oauth/access_token", {
    client_id: options.clientId,
    client_secret: options.clientSecret,
    code: options.code,
    redirect_uri: options.redirectUrl
  });
  const authentication = {
    clientType: options.clientType,
    clientId: options.clientId,
    clientSecret: options.clientSecret,
    token: response.data.access_token,
    scopes: response.data.scope.split(/\s+/).filter(Boolean)
  };
  if (options.clientType === "github-app") {
    if ("refresh_token" in response.data) {
      const apiTimeInMs = new Date(response.headers.date).getTime();
      authentication.refreshToken = response.data.refresh_token, authentication.expiresAt = toTimestamp(apiTimeInMs, response.data.expires_in), authentication.refreshTokenExpiresAt = toTimestamp(apiTimeInMs, response.data.refresh_token_expires_in);
    }
    delete authentication.scopes;
  }
  return {
    ...response,
    authentication
  };
}
function toTimestamp(apiTimeInMs, expirationInSeconds) {
  return new Date(apiTimeInMs + expirationInSeconds * 1000).toISOString();
}

async function createDeviceCode(options) {
  const request$1 = options.request || /* istanbul ignore next: we always pass a custom request in tests */
  request.request;
  const parameters = {
    client_id: options.clientId
  };
  if ("scopes" in options && Array.isArray(options.scopes)) {
    parameters.scope = options.scopes.join(" ");
  }
  return oauthRequest(request$1, "POST /login/device/code", parameters);
}

async function exchangeDeviceCode(options) {
  const request$1 = options.request || /* istanbul ignore next: we always pass a custom request in tests */
  request.request;
  const response = await oauthRequest(request$1, "POST /login/oauth/access_token", {
    client_id: options.clientId,
    device_code: options.code,
    grant_type: "urn:ietf:params:oauth:grant-type:device_code"
  });
  const authentication = {
    clientType: options.clientType,
    clientId: options.clientId,
    token: response.data.access_token,
    scopes: response.data.scope.split(/\s+/).filter(Boolean)
  };
  if ("clientSecret" in options) {
    authentication.clientSecret = options.clientSecret;
  }
  if (options.clientType === "github-app") {
    if ("refresh_token" in response.data) {
      const apiTimeInMs = new Date(response.headers.date).getTime();
      authentication.refreshToken = response.data.refresh_token, authentication.expiresAt = toTimestamp$1(apiTimeInMs, response.data.expires_in), authentication.refreshTokenExpiresAt = toTimestamp$1(apiTimeInMs, response.data.refresh_token_expires_in);
    }
    delete authentication.scopes;
  }
  return {
    ...response,
    authentication
  };
}
function toTimestamp$1(apiTimeInMs, expirationInSeconds) {
  return new Date(apiTimeInMs + expirationInSeconds * 1000).toISOString();
}

async function checkToken(options) {
  const request$1 = options.request || /* istanbul ignore next: we always pass a custom request in tests */
  request.request;
  const response = await request$1("POST /applications/{client_id}/token", {
    headers: {
      authorization: `basic ${btoa(`${options.clientId}:${options.clientSecret}`)}`
    },
    client_id: options.clientId,
    access_token: options.token
  });
  const authentication = {
    clientType: options.clientType,
    clientId: options.clientId,
    clientSecret: options.clientSecret,
    token: options.token,
    scopes: response.data.scopes
  };
  if (response.data.expires_at) authentication.expiresAt = response.data.expires_at;
  if (options.clientType === "github-app") {
    delete authentication.scopes;
  }
  return {
    ...response,
    authentication
  };
}

async function refreshToken(options) {
  const request$1 = options.request || /* istanbul ignore next: we always pass a custom request in tests */
  request.request;
  const response = await oauthRequest(request$1, "POST /login/oauth/access_token", {
    client_id: options.clientId,
    client_secret: options.clientSecret,
    grant_type: "refresh_token",
    refresh_token: options.refreshToken
  });
  const apiTimeInMs = new Date(response.headers.date).getTime();
  const authentication = {
    clientType: "github-app",
    clientId: options.clientId,
    clientSecret: options.clientSecret,
    token: response.data.access_token,
    refreshToken: response.data.refresh_token,
    expiresAt: toTimestamp$2(apiTimeInMs, response.data.expires_in),
    refreshTokenExpiresAt: toTimestamp$2(apiTimeInMs, response.data.refresh_token_expires_in)
  };
  return {
    ...response,
    authentication
  };
}
function toTimestamp$2(apiTimeInMs, expirationInSeconds) {
  return new Date(apiTimeInMs + expirationInSeconds * 1000).toISOString();
}

async function scopeToken(options) {
  const {
    request: optionsRequest,
    clientType,
    clientId,
    clientSecret,
    token,
    ...requestOptions
  } = options;
  const request$1 = optionsRequest || /* istanbul ignore next: we always pass a custom request in tests */
  request.request;
  const response = await request$1("POST /applications/{client_id}/token/scoped", {
    headers: {
      authorization: `basic ${btoa(`${clientId}:${clientSecret}`)}`
    },
    client_id: clientId,
    access_token: token,
    ...requestOptions
  });
  const authentication = Object.assign({
    clientType,
    clientId,
    clientSecret,
    token: response.data.token
  }, response.data.expires_at ? {
    expiresAt: response.data.expires_at
  } : {});
  return {
    ...response,
    authentication
  };
}

async function resetToken(options) {
  const request$1 = options.request || /* istanbul ignore next: we always pass a custom request in tests */
  request.request;
  const auth = btoa(`${options.clientId}:${options.clientSecret}`);
  const response = await request$1("PATCH /applications/{client_id}/token", {
    headers: {
      authorization: `basic ${auth}`
    },
    client_id: options.clientId,
    access_token: options.token
  });
  const authentication = {
    clientType: options.clientType,
    clientId: options.clientId,
    clientSecret: options.clientSecret,
    token: response.data.token,
    scopes: response.data.scopes
  };
  if (response.data.expires_at) authentication.expiresAt = response.data.expires_at;
  if (options.clientType === "github-app") {
    delete authentication.scopes;
  }
  return {
    ...response,
    authentication
  };
}

async function deleteToken(options) {
  const request$1 = options.request || /* istanbul ignore next: we always pass a custom request in tests */
  request.request;
  const auth = btoa(`${options.clientId}:${options.clientSecret}`);
  return request$1("DELETE /applications/{client_id}/token", {
    headers: {
      authorization: `basic ${auth}`
    },
    client_id: options.clientId,
    access_token: options.token
  });
}

async function deleteAuthorization(options) {
  const request$1 = options.request || /* istanbul ignore next: we always pass a custom request in tests */
  request.request;
  const auth = btoa(`${options.clientId}:${options.clientSecret}`);
  return request$1("DELETE /applications/{client_id}/grant", {
    headers: {
      authorization: `basic ${auth}`
    },
    client_id: options.clientId,
    access_token: options.token
  });
}

exports.VERSION = VERSION;
exports.checkToken = checkToken;
exports.createDeviceCode = createDeviceCode;
exports.deleteAuthorization = deleteAuthorization;
exports.deleteToken = deleteToken;
exports.exchangeDeviceCode = exchangeDeviceCode;
exports.exchangeWebFlowCode = exchangeWebFlowCode;
exports.getWebFlowAuthorizationUrl = getWebFlowAuthorizationUrl;
exports.refreshToken = refreshToken;
exports.resetToken = resetToken;
exports.scopeToken = scopeToken;
//# sourceMappingURL=index.js.map
