/******************************************************************************
 *
 *		Copyright U.S. Robotics, 1995, all rights reserved.
 *
 *  PROGRAM: 		iscdevex.h
 *
 *  DESCRIPTION:	Device specific extensions in TAPI for the
 *					PROD_TM.
 *
 *	The first part of this file details the command  structure which
 *	is passed to the TAPI lineDevSpecific() function call. After that,
 *	definitions for various structures are specified. Following the 
 *	structure definitions are the ISC specific events which can be 
 *	reported through TAPI.
 *
 *	Modification Log:
 *
 *	Date		Authors	Notes/Description of Change
 *  06-18-95	JLM		Genesis
 *
 *  12-14-95 	JLM		Changed extension version to 0x00010002 to cover new 
 *						features. 
 *	
 *						Removed the following which were not implemented in the
 *						drivers:
 *
 *						ISC event ISCEVENT_OUTOFSERVICECAUSE.
 *						ISCEVENT_OUTOFSERVICE cause codes:
 *							ISCOOS_NOTINITED, 
 *							ISCOOS_SPIDFALIED,
 *							ISCOOS_PROTOCOLERR.
 *
 *						Changed meaning for loWater and highWater in the 
 *						ISCCALLSPECIFIC structure.
 *
 * 06-12-96		JLM		Chanaged extension version to 0x00010003 to cover new
 *						features.
 * 
 *						Added new commands:
 *							ISCCMD_SET_SPECIAL_ACCESS_CODE
 *						Addedd new board features:
 *							ISCBOARDFEATURE_DATAOVERVOICE
 * 
 * 10-14-96		RZW		Added new boardfeature support applicable to INS 64
 *						switch type.
 *
 * 12-02-96		JLM		Found that CALLDEVSPECIFIC events were not being 
 *						reported by TAPI. Have changed to send 
 *						CALLINFOSTATE_OTHER events with dwParam2 set to the
 *						device specific event we are sending.
 *
 *						Added new commands to manually bring up and down a 
 *						second link in a PPP bundle when using Turbo.
 *
 *						Added field to specify link expansion method.
 * 03-03-97		JLM		Added new call info event to send our call handle
 *						to be used in subsequent ioctl commands.
 *****************************************************************************/

#ifndef _iscdevext_h_included
#define _iscdevext_h_included

#ifndef _iscconst_h_included
#include "iscconst.h"
#endif

/*****
*  Ensure all programmers are using same byte-alignment for structs.
*****/
#pragma pack(4)

#ifndef TAPIMAXDESTADDRESSSIZE
#define TAPIMAXDESTADDRESSSIZE	80L
#endif

#ifndef TAPIMAXDEVICECLASSSIZE
#define TAPIMAXDEVICECLASSSIZE	40L
#endif

// ***********************************************************************
// This is the extended version number which should be used to
// negotiate the features described in this header file.
// ***********************************************************************
#define ISC_EXT_VERSION		0x00010003		

// ***********************************************************************
// This is the device specific extension structure which is passed
// as the argument to lpParams in the lineDevSpecific() call. This is 
// the function that is used to enable ISC specific features. The other
// arguments to lineDevSpecific() have meaning depending on the command
// issued.
// ***********************************************************************

typedef struct tagISCDEVSPECIFIC
	{
	DWORD	dwCommand;	// Device Specific Feature to activate.
	LPVOID  lpvArgs;	// Feature specific arguments or NULL.
	} ISCDEVSPECIFIC, FAR *LPISCDEVSPECIFIC;


// ***********************************************************************
// The following are the individual commands that can be sent in uCommand
// of the ISCDEBSPECIFIC structure and their related arguments.
// ***********************************************************************

#define	ISC_APP_COMMAND_BASE	100

/*
// Reset board drops all current calls and will reset level one and two
//  and then re-aquire with the switch. This command WILL effect all
//  open line devices, not just the one referenced. It most useful
//  when you have a received a LINEDEVSTATE_DISCONNECTED or OUTOFSERVICE
//  message which could have been caused by a bad board setting, SPID,
//  and so forth.
//
//  lineDevSpecific
//			(
//			hLine 		- Set to line device for the board to reset.	
//          dwAddressId	- n/a.
//			hCall		- n/a.
//			lpParams 	- LPISCDEVSPECIFIC.
//			dwSize		- sizeof (ISCDEVSPECIFIC).
//			)
//
//  dwCommand	- ISC_BOARD_RESET.
//  lpvArgs  	- Points to a valid ISCBRICONFIG structure.
*/
#define	ISCCMD_BOARD_RESET			(ISC_APP_COMMAND_BASE + 0)


/*
// This command retrieves statistics for the call referenced.  
//
//  lineDevSpecific
//			(
//			hLine 		- Set to line device for the call referenced.
//          dwAddressId	- n/a.
//			hCall		- Call to get stats for. Must be connected.
//			lpParams 	- LPISCDEVSPECIFIC.
//			dwSize		- sizeof (ISCDEVSPECIFIC).
//			)
//
//  dwCommand	- ISC_GET_STATS.
//  lpvArgs  	- Points to a ISCSTATSDATA structure.
*/
#define	ISCCMD_GET_STATS			(ISC_APP_COMMAND_BASE + 1)


/*
// Get board configuration returns the current board configuration.
// 
//  lineDevSpecific
//			(
//			hLine 		- Set to line device for the board config
//						  you want.
//          dwAddressId	- n/a.
//			hCall		- n/a.
//			lpParams 	- LPISCDEVSPECIFIC.
//			dwSize		- sizeof (ISCDEVSPECIFIC).
//			)
//
//  dwCommand	- ISC_GET_BOARD_CONFIGURATION.
//  lpvArgs  	- Points to a block of memory large enough to hold
//				  a ISCBRICONFIG structure. The data will be copied into
//				  this location.
*/
#define	ISCCMD_GET_BOARD_CONFIGURATION		(ISC_APP_COMMAND_BASE + 2)

/*
// Sets the default incoming protocol and line features to use for the 
// refernced line when answering a call. The default is TRANSPARENT. 
//
//  lineDevSpecific
//			(
//			hLine 		- Set to line device.
//          dwAddressId	- n/a.
//			hCall		- n/a.
//			lpParams 	- LPISCDEVSPECIFIC.
//			dwSize		- sizeof (ISCDEVSPECIFIC).
//			)
//
//  dwCommand	- ISC_SET_CALLDEFAULT.
//  lpvArgs  	- Points to a valid ISCCALLSPECIFIC structure.
//
// The encapsulation parameters can be retrieved on a call basis by doing a 
//  lineGetcallInfo() and then retrieving the ISCCALLSPECIFIC structure
//  from the dwDevSpecificOffset. They can be retrieved on a line basis 
//  by makeing a call to lineGetDevCaps() and lineGetDevStatus() and 
//  retrieving the ISCCALLSPECIFIC structure from the dwDevSpecificOffset.
//
// This command will generate a LINEDEVSTATE_DEVSPECIFIC event.
*/
#define	ISCCMD_SET_CALLDEFAULT	(ISC_APP_COMMAND_BASE + 3)

/*
// Sets the line features to use for makeing and receiving calls on
// this line device.
//
//  lineDevSpecific
//			(
//			hLine 		- Set to any line device.
//          dwAddressId	- n/a.
//			hCall		- n/a.
//			lpParams 	- LPISCDEVSPECIFIC.
//			dwSize		- sizeof (ISCDEVSPECIFIC).
//			)
//
//  dwCommand	- ISCCMD_SET_BOARDFEATURES.
//  lpvArgs  	- Points to a valid DWORD which contains the 
//					ISCBOARDFEATURES (constants) to use.
//
//  They can be retrieved on a line basis 
//  by makeing a call to lineGetDevCaps() and lineGetDevStatus() and 
//  retrieving the ISCCALLSPECIFIC structure from the dwDevSpecificOffset.
//
// This command will generate a LINEDEVSTATE_DEVSPECIFIC event.
*/
#define	ISCCMD_SET_BOARDFEATURES	(ISC_APP_COMMAND_BASE + 4)

/*
//  Stop drops all current calls and will cause the device to disconnect 
//  from the switch.
//
//  This command WILL effect all open line devices, not just the 
//  one referenced. 
//
//  lineDevSpecific
//			(
//			hLine 		- Set to line device for the board to stop.	
//          dwAddressId	- n/a.
//			hCall		- n/a.
//			lpParams 	- LPISCDEVSPECIFIC.
//			dwSize		- sizeof (ISCDEVSPECIFIC).
//			)
//
//  dwCommand	- ISC_BOARD_STOP.
//  lpvArgs  	- NULL.
*/
#define	ISCCMD_BOARD_STOP			(ISC_APP_COMMAND_BASE + 5)

/*
// Runs board level diagnostics and returns the status of that
//  operation.
//
//  lineDevSpecific
//			(
//			hLine 		- Set to any line device.
//          dwAddressId	- n/a.
//			hCall		- n/a.
//			lpParams 	- LPISCDEVSPECIFIC.
//			dwSize		- sizeof (ISCDEVSPECIFIC).
//			)
//
//  dwCommand	- ISCCMD_RUNBOARDDIAGS
//  lpvArgs  	- Points to a valid ISCDIAGNOSTIC structure.
//
// This command will generate a LINEDEVSTATE_DEVSPECIFIC event.
*/
#define	ISCCMD_RUNBOARDDIAGS	(ISC_APP_COMMAND_BASE + 6)

/*
// Generate a board ringing pattern.
//
//  lineDevSpecific
//			(
//			hLine 		- Set to any line device.
//          dwAddressId	- n/a.
//			hCall		- n/a.
//			lpParams 	- LPISCDEVSPECIFIC.
//			dwSize		- sizeof (ISCDEVSPECIFIC).
//			)
//
//  dwCommand	- ISCCMD_RINGBOARD
//  lpvArgs  	- Points to a valid ISCDIAGNOSTIC structure.
//
// This command will generate a LINEDEVSTATE_DEVSPECIFIC event.
*/
#define	ISCCMD_RINGBOARD	(ISC_APP_COMMAND_BASE + 7)

/*
// Sets the defaults for TurboPPP if the TurboPPP (ML Encapsulation)
// board feature is set. Use of this structure allows you to tailor
// TurboPPP.
//
//  lineDevSpecific
//			(
//			hLine 		- Set to line device.
//          dwAddressId	- n/a.
//			hCall		- n/a.
//			lpParams 	- LPISCDEVSPECIFIC.
//			dwSize		- sizeof (ISCDEVSPECIFIC).
//			)
//
//  dwCommand	- ISCCMD_SET_TURBOPPP_SPECIFIC
//  lpvArgs  	- Points to a valid ISCCALLSPECIFICDATA structure.
//
// This command will generate a LINEDEVSTATE_DEVSPECIFIC event.
//
// This command new for extension version 0x00010003.		
*/
#define	ISCCMD_SET_TURBOPPP_SPECIFIC	(ISC_APP_COMMAND_BASE + 8)

/*
// Sets special access codes that may be needed to dial a digital
// and/or analog call. Special codes are those that meet the following
// criteria:
//
//    1. The access codes to dial are different for voice and data calls.
//			That is, a #9 may be needed for data and *9 needed for voice.
//
//  lineDevSpecific
//			(
//			hLine 		- Set to line device.
//          dwAddressId	- n/a.
//			hCall		- n/a.
//			lpParams 	- LPISCDEVSPECIFIC.
//			dwSize		- sizeof (ISCDEVSPECIFIC).
//			)
//
//  dwCommand	- ISCCMD_SET_SPECIAL_ACCESS_CODE
//  lpvArgs  	- Points to a valid ISCACCESSCODE structure.
//
// This command will generate a LINEDEVSTATE_DEVSPECIFIC event.
//
// This command new for extension version 0x00010003.		
*/
#define	ISCCMD_SET_SPECIAL_ACCESS_CODE	(ISC_APP_COMMAND_BASE + 9)


/*
// Gets the special access codes that may be in use on the line. 
//
//  lineDevSpecific
//			(
//			hLine 		- Set to line device.
//          dwAddressId	- n/a.
//			hCall		- n/a.
//			lpParams 	- LPISCDEVSPECIFIC.
//			dwSize		- sizeof (ISCDEVSPECIFIC).
//			)
//
//  dwCommand	- ISCCMD_SET_SPECIAL_ACCESS_CODE
//  lpvArgs  	- Points to a valid ISCACCESSCODE structure.
//
// This command will generate a LINEDEVSTATE_DEVSPECIFIC event.
//
// This command new for extension version 0x00010003.		
*/
#define	ISCCMD_GET_SPECIAL_ACCESS_CODE	(ISC_APP_COMMAND_BASE + 10)

/*
// Bring up a second link in a TurboPPP PPP multi-link bundle.
//
//  lineDevSpecific
//			(
//			hLine 		- Set to line device.
//          dwAddressId	- n/a.
//			hCall		- Call that link should be raised in.
//			lpParams 	- NULL
//			dwSize		- 0
//			)
//
//  dwCommand	- ISCCMD_RAISE_TURBOPPP_LINK
//  lpvArgs  	- Points to a valid ISCACCESSCODE structure.
//
// This command will generate a LINEDEVSTATE_DEVSPECIFIC event.
//
// This command new for extension version 0x00010003.		
*/
#define	ISCCMD_RAISE_TURBOPPP_LINK	(ISC_APP_COMMAND_BASE + 11)

/*
// Bring down a second link in a TurboPPP PPP multi-link bundle.
//
//  lineDevSpecific
//			(
//			hLine 		- Set to line device.
//          dwAddressId	- n/a.
//			hCall		- Call that link should be raised in.
//			lpParams 	- NULL
//			dwSize		- 0
//			)
//
//  dwCommand	- ISCCMD_LOWER_TURBOPPP_LINK
//  lpvArgs  	- Points to a valid ISCACCESSCODE structure.
//
// This command will generate a LINEDEVSTATE_DEVSPECIFIC event.
//
// This command new for extension version 0x00010003.		
*/
#define	ISCCMD_LOWER_TURBOPPP_LINK	(ISC_APP_COMMAND_BASE + 12)

// ***********************************************************************
// The following data structures and constants are used with 
// the commands above.			
// ***********************************************************************

// ***********************************************************************
//					ISCBOARDFEATURE CONSTANTS
//
//	A 0 in that bit position means a feature is diabled, while a 1 
//  indicates it is enabled. These features effect the behavior
//  of all line devices on the card for the handling of calls.
// ***********************************************************************

// If VOICEOVERRIDE is enabled, any voice call will cause a second 
//  data call to be dropped so that the call can be made.
#define	ISCBOARDFEATURE_VOICEOVERRIDE	  	0x00000001

// If ACCEPTVOICEASDATA is enabled, any call into the line device
//  with a voice type bearer mode will be treated as a data call 
//  instead of an inbound analog phone call. This allows devices
//  to make data calls using a voice bearer capability.
#define	ISCBOARDFEATURE_ACCEPTVOICEASDATA	0x00000002

// If BEEPONDATA is enabled, once a data call is connected on the 
//  line device, the driver will beep the speaker.
#define	ISCBOARDFEATURE_BEEPONDATA	 		0x00000004

// If PPPEMULATION is enabled, all data calls will be subject to 
//  TurboPPP
#define	ISCBOARDFEATURE_PPPEMULATION		0x00000008

// If MPPCPNEGOTIATE is enabled, and the other side can do
//  multilink we will negotiate the control protocols using 
//  a multi link header.
// This board feature new for extension version 0x00010002.
#define	ISCBOARDFEATURE_MPPCPNEGOTIATE		0x00000010

// If INCOMINGONDIALUP is enabled, then all incoming digital
//  data calls will "appear" on the dialup networking line
//  device.
// This board feature new for extension version 0x00010002.
#define	ISCBOARDFEATURE_INCOMINGONDIALUP	0x00000020

// Same as PASSTHRUMODE_QUEUEPRINTS in iscconst.h. If this
// bit is set then we do not allow the port to be shared.
// This flag is provided here for dynamically setting
// a share versus no share option.
// This board feature new for extension version 0x00010002.
#define	ISCBOARDFEATURE_QUEUEPRINTS			0x00000040

// If this bit is set then all outbound data calls will be 
// made using a Voice bearer mode.
// This board feature new for extension version 0x00010003.		
#define	ISCBOARDFEATURE_DATAOVERVOICE		0x00000080 

// if this bit is set according to the config setting, the
// phone line will apply call restriction. This feature applies
// to the INS 64 switch type.
#define ISCBOARDFEATURE_CALLRESTRICTION		0x00000100	


// ***********************************************************************
//					ISCLINEFEATURE CONSTANTS
//
//	A 0 in that bit position means a feature is diabled, while a 1 
//  indicates it is enabled. These are per call features.
// ***********************************************************************

// If 56RETRY is enabled, any call which fails to connect at 64000kbps
//  will automaticall be re-tried at 56000kbps.
#define	ISCLINEFEATURE_56RETRY			0x00000001


// ***********************************************************************
// This is the Access Code strucutre used with the 
//  ISCCMD_SET_SPECIAL_ACCESS_CODE and
//  ISCCMD_GET_SPECIAL_ACCESS_CODE command.
// This structure new for extension version 0x00010003.		
// ***********************************************************************
typedef struct tagISCACCESSCODE 
	{
    DWORD		dwTotalSize;	// Set to size of this structure
    DWORD		dwNeededSize;	// Set by driver if more space needed 
								//  to fulfill request
    DWORD		dwUsedSize;	    // Set by driver to amount of data returned.

				// Access code for Voice calls.
	char		szVoiceAccessCode[MAXACCESSCODESIZE]; 

				// Access code for Data calls.
	char		szDataAccessCode[MAXACCESSCODESIZE];	 
	} ISCACCESSCODE, FAR *LPISCACCESSCODE;

// ***********************************************************************
// This is the ISC DIAGNOSTIC structure. This strucutre is used when
// diagnostics are to be run on the Adapter. 
// ***********************************************************************
typedef struct tagISCDIAGNOSTIC
	{
    DWORD		dwTotalSize;		// Set to size of this structure
    DWORD		dwNeededSize;		// Set by driver if more space needed to 
									//  fulfill request
    DWORD		dwUsedSize;		    // Set by driver to amount of data returned.
  	DWORD		dwIOBase;			// Current board I/O Address
  	DWORD		dwIRQ;				// Current board IRQ
	DWORD		dwRingPattern;		// Pattern to ring (1-8)
	DWORD		dwErrorReturn;		// Result of running diagnostic. See
									//  SLF_xxxx constants in iscconst.h
	DWORD		dwBoardType;		// Board type (SIB_EXTERNAL_BOARD or 
									//  SIB_INTERNAL_BOARD)
    DWORD		dwParallelPortMode;	// Parallel port type.
  									// device name for parallel port
  	char		szPortName[MAXPORTNAMESIZE]; 
	DWORD		dwReserved1;		// not used - future expansion.
	DWORD		dwReserved2;		// not used - future expansion.
	} ISCDIAGNOSTIC, FAR *LPISCDIAGNOSTIC;

// ***********************************************************************
// This is the ISC Call Specific data structure used on a
// lineMakeCall or returned on a lineGetCallInfo. It documents the
// protocol encapsulation in effect or invokes it.
// ***********************************************************************
typedef struct tagISCCALLSPECIFICDATA
	{
    DWORD	dwTotalSize;		// Set to size of this structure
    DWORD	dwNeededSize;		// Set by driver if more space needed to 
								//  fulfill request
    DWORD	dwUsedSize;		    // Set by driver to amount of data returned.
	DWORD	dwEncapsulation;	// Protocol encapsulation in use.
								//   Use Call Encapsulation constants
  	UINT	protocols;			// PPP protocols to use. Should be OR'd
								// together to get what you want. If a
								// location is zero, that protocol is not used.
								//   Use PPP Protocol constants

	// The IP and IPX addresses are encoded as:
	//  ddd.ddd.ddd.ddd where ddd is one byte in the int.

  	DWORD	ipAddress;			// IP address if known or set to
								// zero (0) to have it negotiated.

  	DWORD	ipxNetAddress;		// IPX Network Address if needed.

  	DWORD	compression;		// Use Compression constants

	// The following constants are used for dynamic bandwidth 
	// expansion when using our PPP/ML or PPP/ML  EMULATION (TurboPPP).
	// If traffic over the link exceeds the percentage specified by 
	// the highWater for the time period specified by loWater then 
	// a second link is brought up.
	// 
	DWORD	dwHighWater;		// Specifies percentage of line utilization that
								// must occur before next B channel is allocated
								// to the connection.  Range = 0 thru 100%;
								// (0% means next B channel is always allocated,
								// 100% means that NO dynamic allocation will
								// take place)
	
	DWORD	dwSamplingHistory;	// Specifies the number of seconds of sampling 
								// history used to determine current throughput 
								// rates.  Range: 12 - 600 seconds, default = 30
								// (TPSH_LOWER, TPSH_UPPER, and TPSH_DEFAULT 
								// defined in iscconst.h specify these values)

  	DWORD	dwUseTurboPPPCompression;	// T = use compression, F = dont

	DWORD	dwUnexpandTime;		// ... (to be defined later)

	// Password for authentication if using PAP or CHAP.
  	char	password[MAXSLPASSWDSIZE];

	// User name for authentication if using PAP or CHAP.
  	char	username[MAXSLUSERNAMESIZE];

	// NetBIOS name
  	char	netbiosName[MAXSLNBNAMESIZE];

	// Second phone number. Needed when using the PPML_ emulation
	//  or encapsulation framing. Specifies the number to dial to
	//  bring up the second call. Must be specified, even if it 
	//  is the same as the first.
  	char	phoneNumber[TAPIMAXDESTADDRESSSIZE];	

	// The combination of ISCLINEFEATURE_ constants to use or in
	//  use on the call.
	DWORD	dwLineFeatures;

	// This field receives the current combination of ISCBOARDFEATURE_ 
	//  constants which are in effect on the line.  To set board features,
	//  use the ISCCMD_SET_BOARDFEATURES device-specific call.
	DWORD	dwBoardFeatures;

	// These two fields are used to control how we deal with a MLPPP call
	//  that has been split acrross two physical devices. The retyr count
	//  specifies the number of consecutive times we will retry the call 
	//  and the retry time specifies the interval in-between those checks.
	DWORD	dwSplitRetryCount;		
	DWORD	dwSplitRetryTime;	

	// Specifies link expansion method when using TurboPPP. See constants
    // file for definitions.
	DWORD	dwLinkExpansionMethod;

	}
	ISCCALLSPECIFICDATA, FAR *LPISCCALLSPECIFICDATA;

// ***********************************************************************
// This is the statistics structure where statistics info is copied (on 
// a per call basis).  NOTE: The following items are gotten from the
// following structs using the TAPI lineGetCallInfo func:
//
// 	CALL SPECIFIC struct:
//		- call protocol
//	 	- compression algorithm
//
// 	CALL INFO:
//		- speed
// 	 	- direction
// 	 	- phone numbers
// ***********************************************************************

typedef struct tagISCSTATSDATA 
  {
  DWORD	dwTotalSize;		// Set to size of this structure
  DWORD	dwNeededSize;		// Set by driver if more space needed to fulfill request
  DWORD	dwUsedSize;			// Set by driver to amount of data returned.

  DWORD	conn_secs;			// Seconds call connected
  DWORD	conn_tx_frames;		// Total TX frames for call
  DWORD	conn_tx_bytes;		// Total TX bytes for call
  DWORD	conn_rx_frames;		// Total RX frames for call
  DWORD	conn_rx_bytes;		// Total RX bytes for call
  DWORD board_rx_tiny;		// Total Rx msg too small count
  DWORD board_rx_big;		// Total Rx msg too large count
  DWORD board_tx_tmo;		// Total Tx timeout count
  DWORD board_tx_big;		// Total Tx packet too big count
  DWORD board_tx_fail;		// Total Tx failure count
  DWORD board_tx_eom;		// Total Tx end of msg count
  DWORD	board_tx_urun;		// Total TX underruns for call
  DWORD	board_rx_orun;		// Total RX overruns for call
  DWORD	board_rx_crc_err;	// Total RX CRC errors for call
  DWORD dchn_rx_tiny;		// D Channel -Rx msg too small count
  DWORD dchn_rx_big;		// D Channel -Rx msg too large count
  DWORD dchn_rx_anomalies;	// D Channel -Unexpected Rx H/W 
  							//    status/states
  DWORD dchn_tx_tmo;		// D Channel -Tx timeout count
  DWORD dchn_rx_byte_cnt;	// D Channel -Rx byte count
  DWORD dchn_tx_byte_cnt;	// D Channel -Tx byte count

  DWORD	dwCompressErrorCount; // Number of compression/de-compression errors.
  DWORD	dwBadSeqNbrCount;     // Number times sequence numbers in stac 
							  //  compression were not correct.
  DWORD	dwTotalSplitRetries;  // Number of times we split physical devices 
							  //  during the life of the call.
  }
  ISCSTATSDATA, FAR *LPISCSTATSDATA;

/**********************************************************************
*		Definitions to use with the RESET BOARD command.
***********************************************************************/

/*****
*  ISCCHANCONFIG (channel) config struct
*****/
typedef struct tagISCCHANCONFIG 
  {
  unsigned
  char		szPhoneNum[TAPIMAXDESTADDRESSSIZE];	// Phone number for B channel
  unsigned
  char		szSPID[TAPIMAXDESTADDRESSSIZE];		// SPID for B channel
  }	
  ISCCHANCONFIG, FAR *LPISCCHANCONFIG;

/*****
*  Board configuration struct
*****/

typedef struct tagISCBRICONFIG 
  {
  DWORD	dwTotalSize;		// Set to size of this structure
  DWORD	dwNeededSize;		// Set by driver if more space needed to fulfill request
  DWORD	dwUsedSize;			// Set by driver to amount of data returned.

  DWORD			dwTelcoSwitch;		// (Use Switch Type constants)
  DWORD			dwIOBase;			// Board I/O Address
  DWORD			dwIRQ;				// Board IRQ

  									// This shoudl be one or two based upon
  									// the number of phone numbers in use.
									// This controls access to the chan
									// configuration. That is, if it is 
									// a one then only chan[0] is valid.
									// If it is a two then chan[0] and chan[1]
									// are valid.

  DWORD			dwBoardType;		// Board type (SIB_EXTERNAL_BOARD or 
  									//             SIB_INTERNAL_BOARD)

  DWORD			dwParallelPortMode;	// Parallel port type (4-bit , EPP, ECP
  									//   or 8 bit ).


  DWORD			dwPassThruMode;		// How we should handle pass through
									//  See PASSTHRUMODE_ constants in
									//  iscconst.h.

  					// device name for parallel port
  char			szPortName[MAXPORTNAMESIZE]; 

  									// B Channel info for card
  DWORD			dwNumPhoneNumbers;	
  ISCCHANCONFIG	chan[MAXSLBRICHANS];

  									// True if board is a U interface,
									//  else it is a S/T.
									//  This field is read only.
  BOOL			fIsUTermination;


  									// True if a data only board.
									//  This field is read only.
  BOOL			fDataOnly;			

									// True if board supports external
									// ring generator.
									//  This field is read only.
  BOOL			fExternalRing;

  									// Cards serial number burned
  									// into eeprom. This is NULL 
									// terminated. 
									//  This field is read only.
  unsigned
  char			szSerNumber[MAXSERIALNUMBERSIZE];

  									// Cards MAC address burned
  									// into eeprom. This is NOT
									// NULL terminated. 
									//  This field is read only.
  unsigned
  char 			ucMacAddress[MACADDRESSSIZE];

  DWORD	dwReserved1;		// not used - future expansion.
  DWORD	dwReserved2;		// not used - future expansion.
  }	
  ISCBRICONFIG, FAR *LPISCBRICONFIG;

// ***********************************************************************
// The following are SecureLink Device Specific Events. These events will
// be reported using the LINE_DEVSPECIFIC TAPI Message Type.
// ***********************************************************************

/*
// This event may sent whenever a call terminates with an UNKNOWN
//  disconnect mode which was caused by receiving a specific ISDN cause
//  code that could not be translated to a TAPI disconnect mode.
//
// hdLine:    Line device event occurred on.
// htCall:    Call that is using compression.
// dwMsg:     LINE_CALLINFO
// dwParam1:  LINECALLINFOSTATE_OTHER
// dwParam2:  ISCEVENT_ISDNCAUSECODE
// dwParam3:  ISDN cause code received from the network. See CCITT spec.
*/
#define	ISCEVENT_ISDNCAUSECODE	  0x00000001


/*
// This message is sent if the peer in PPP does not support
//  compression.
//
// hdLine:    Line device event occurred on.
// htCall:    PPP Call we tried to negotiate compression for.
// dwMsg:     LINE_CALLINFO
// dwParam1:  LINECALLINFOSTATE_OTHER
// dwParam2:  ISCEVENT_COMPNOTENABLED
// dwParam3:  n/a
*/
#define	ISCEVENT_COMPNOTENABLED			0x00000002

/*
// This message is sent if the peer in PPP does not support
//   a particulatr protocol and we are disabling it.
//
// hdLine:    Line device event occurred on.
// htCall:    PPP Call we tried to negotiate compression for.
// dwMsg:     LINE_CALLINFO
// dwParam1:  LINECALLINFOSTATE_OTHER
// dwParam2:  ISCEVENT_PROTOCOLDISABLED
// dwParam3:  Protocol that was disabled. See PPP_PROT_ constants.
*/
#define	ISCEVENT_PROTOCOLDISABLED		0x00000003

/*
// This event is sent whenever a PPP link has gone down
//  unexpectedly and we have re-started it.
//
// hdLine:    Line device event occurred on.
// htCall:    PPP call.
// dwMsg:     LINE_CALLINFO
// dwParam1:  LINECALLINFOSTATE_OTHER
// dwParam2:  ISCEVENT_PPPRESTART
// dwParam3:  n/a
*/
#define	ISCEVENT_PPPRESTART	  		0x00000004

/*
// This message is sent after CHAP has authenticated the user.
//
// hdLine:    Line device event occurred on.
// htCall:    PPP Call.
// dwMsg:     LINE_CALLINFO
// dwParam1:  LINECALLINFOSTATE_OTHER
// dwParam2:  ISCEVENT_AUTHENTICATED
// dwParam3:  TRUE (expanded link in a ML environment).
//			  FALSE(first link in  a ML environment or a PPP link).
*/
#define	ISCEVENT_AUTHENTICATED	0x00000006

/*
// These messages are sent whenever a PPP protocol is negotiated.
//  For some protocols, if negotiated, the application can do a 
//  lineGetCallInfo() to retrieve further information about the call. 
//
// hdLine:    Line device event occurred on.
// htCall:    PPP Call.
// dwMsg:     LINE_CALLINFO
// dwParam1:  LINECALLINFOSTATE_OTHER
// dwParam2:  ISCEVENT_PROTOCOLxx
// dwParam3:  See below.
//
*/
#define	ISCEVENT_PROTOCOLPAP	0x00000007	// Password Authentication

#define	ISCEVENT_PROTOCOLCPAP	0x00000008	// Challenege Authentication

#define	ISCEVENT_PROTOCOLBR  	0x00000009	// Bridging
// dwParam2:  n/a
// dwParam3:  n/a

#define	ISCEVENT_PROTOCOLIPX  	0x00000010	// IPX
// Application can do a lineGetCallInfo() to retrieve the mac address
// from the ISCCALLSPECIFIC structure.
// dwParam3:  Contains the IPX address as ddd.ddd.ddd.ddd where ddd is one 
//            byte in the int.

#define	ISCEVENT_PROTOCOLNBF  	0x00000011	// NetBeui
// Application can do a lineGetCallInfo() to retrieve the netbios name
// from the ISCCALLSPECIFIC structure.
// dwParam3:  n/a

#define	ISCEVENT_PROTOCOLCCP  	0x00000012	// Compression Control
// dwParam3 = Compression in use. See compression constants.

#define	ISCEVENT_PROTOCOLIP  	0x00000013	// IP
// dwParam3:  Contains the IP address as ddd.ddd.ddd.ddd where ddd is one 
//            byte in the int. An address of all zeros means that an IP
//			  address was not negotiated.
// We lose this info:  0 - Address was configured, 1 - Address was negotiated.

/*
// This message is sent whenever a PPP connection is terminated
//  for a non-normal reason.
//
// hdLine:    Line device event occurred on.
// htCall:    Call that IP address is negotiated for.
// dwMsg:     LINE_CALLINFO
// dwParam1:  LINECALLINFOSTATE_OTHER
// dwParam2:  ISCEVENT_PPPCAUSE
// dwParam3:  Set to an PPPCAUSE_ constant.
*/
#define	ISCEVENT_PPPCAUSE	0x00000014

/*
//  ISCEVENT_PPPCAUSE cause codes for dwParam2.
*/
#define  PPPCAUSE_CONNECTTMO			0x00000001	// Connection timed out.
#define  PPPCAUSE_AUTHENTICATETMO		0x00000002	// Authenticater has not
													//  replied.
#define  PPPCAUSE_NOPEERAUTHENTICATION	0x00000003	// Peer has not requested 
													//  authentication.
#define  PPPCAUSE_INVALIDCHAPRESPONSE	0x00000004	// Invalid CHAP response.
#define  PPPCAUSE_CHAPTMO            	0x00000005	// CHAP challenge timeout.
#define  PPPCAUSE_CHAPMISMATCH         	0x00000006	// PPP Auth Failed, CHAP 
													//  mismatch. 
#define  PPPCAUSE_PAPMISMATCH         	0x00000007	// PPP Auth Failed, PAP 
													//  mismatch. 


/*
// This message is sent whenever an application asks to use the parallel
//  port on the external but we are not allowing the device to be 
//  shared.
//
// hdLine:    Line device event occurred on.
// htCall:    n/a
// dwMsg:     LINE_CALLINFO
// dwParam1:  LINECALLINFOSTATE_OTHER
// dwParam2:  ISCEVENT_TRIEDTOPRINT
// dwParam3:  n/a
*/
#define	ISCEVENT_TRIEDTOPRINT	0x00000015

/*
// This message is sent once during the life of a call when we detect that
//  a MLPPP call has been split across two physical devices.
//
// hdLine:    Line device event occurred on.
// htCall:    Call  where split occurred
// dwMsg:     LINE_CALLINFO
// dwParam1:  LINECALLINFOSTATE_OTHER
// dwParam2:  ISCEVENT_MLPPPCALLSPLIT
// dwParam3:  n/a
*/
#define	ISCEVENT_MLPPPCALLSPLIT	0x00000016

/*
// This message is sent once a call is connected and passes up the drivers
// handle to the call. This call handle must be used in subsequent
// calls through the IOCTL interface when required.
//
// hdLine:    Line device event occurred on.
// htCall:    Call  where split occurred
// dwMsg:     LINE_CALLINFO
// dwParam1:  LINECALLINFOSTATE_OTHER
// dwParam2:  ISCEVENT_DRIVERHDCALL	
// dwParam3:  handle to the drivers call.
*/
#define	ISCEVENT_DRIVERHDCALL	0x00000017

/*****
*  Revert to the default packing alignment.  NOTE:  For the Borlandc compiler,
*  there is no way to revert back.  Thus, for Borlandc builds, everyone must
*  be using 
*****/
#pragma pack()

#endif	// iscdevext_h_included

